// RetroLevelControlClass.js

var RetroLevelControlClass = {
	name : "RetroLevelControlClass",
	description : "namespace for Gamma-Scientific laserlux_G7 thresholds",
	version : '1.0',
	variableSize : false,
	cntr : 0,
	lvlNames : [ "Ignored", "Valid", "0Marginal", "1Marginal", "2Marginal", "Good" ], // manually encoded marginal 0-2. matches software\Laserlux\src\threshold.h::RETRO_*_KEY and parsed by readThreshold()
	ctrlList : []
};
RetroLevelControlClass.RETRO_THRESHOLD_KEY = "RTHR";
RetroLevelControlClass.RETRO_COLOR_NAMES = [ "White", "Yellow" ];

// CSS 1–2.0, HTML 3.2–4, and VGA color names
// <https://en.wikipedia.org/wiki/Web_colors>
// aka sRGB
RetroLevelControlClass.vga = [];

// -------------------------------------------------------------------------
// static functions
RetroLevelControlClass.init = function() {
	this.ctrlWhite = new RetroLevelControlClass.StripeThresholdClass("White", "white");
	this.ctrlYellow = new RetroLevelControlClass.StripeThresholdClass("Yellow", "yellow");
	this.ctrlList = []; // clear list
	this.ctrlList[this.ctrlList.length] = this.ctrlYellow;
	this.ctrlList[this.ctrlList.length] = this.ctrlWhite;
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("White", 255, 255, 255);
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Silver", 192, 192, 192);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Gray", 128, 128, 128);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Black", 0, 0, 0);
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Red", 255, 0, 0);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Maroon", 128, 0, 0);
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Yellow", 255, 255, 0);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Olive", 128, 128, 0);
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Lime", 0, 255, 0);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Green", 0, 128, 0);
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Aqua", 0, 255, 255);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Teal", 0, 128, 128);
	this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Blue", 0, 0, 255);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Navy", 0, 0, 128);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Fuchsia", 255, 0, 255);
	// this.vga[this.vga.length] = new RetroLevelControlClass.RetroColorClass("Purple", 128, 0, 128);
}

RetroLevelControlClass.drawCtrl = function(node) {
	if (!node)
		return;
	var html = "";
	for (var ndx = 0; ndx < this.ctrlList.length; ndx++) {
		html += this.ctrlList[ndx].getHtml();
	}
	node.innerHTML = html;
	this.addChangeHandlers();
	this.updatePosition();
}

RetroLevelControlClass.addChangeHandlers = function(_callback) {
	for (var ndx = 0; ndx < this.ctrlList.length; ndx++) {
		this.ctrlList[ndx].addChangeHandler(_callback);
	}
	this.updatePosition();
}

RetroLevelControlClass.updatePosition = function() {
	for (var ndx = 0; ndx < this.ctrlList.length; ndx++) {
		this.ctrlList[ndx].updatePosition();
	}
}

RetroLevelControlClass.setCtrlValue = function(obj) {
	var ctrl = this.ctrlWhite;
	if (obj["colorname"] && obj.colorname.toLowerCase() == this.ctrlYellow.name.toLowerCase())
		ctrl = this.ctrlYellow;
	ctrl.copy(obj);
	return ctrl;
}

RetroLevelControlClass.getStripe = function(ndx) {
	return ndx == 1 ? this.ctrlYellow : this.ctrlWhite;
}

// -------------------------------------------------------------------------
// ctrl object
RetroLevelControlClass.StripeThresholdClass = function(_name, _bckcolor) {
	this.name = _name;
	this.bckcolor = "white";
	if (_bckcolor)
		this.bckcolor = _bckcolor;
	this.maxValue = 100.0;
	this.lvlList = [];
	for (var ndx = 0; ndx < RetroLevelControlClass.lvlNames.length; ndx++) {
		this.lvlList[this.lvlList.length] = new RetroLevelControlClass.ThresholdLevelClass(this, RetroLevelControlClass.lvlNames[ndx], this.maxValue * ndx / RetroLevelControlClass.lvlNames.length, ndx == 0 ? RetroLevelControlClass.Black : null);
	}
}

RetroLevelControlClass.StripeThresholdClass.prototype.copy = function(obj) {
	if (obj["colorname"] && obj.colorname.toLowerCase() == this.name.toLowerCase()) {
		if (obj["maxvalue"])
			this.maxValue = obj.maxvalue;
		if (obj["label"])
			this.label = obj.label;
		if (obj["children"] && obj.children.length) {
			for (var ndx = 0; ndx < obj.children.length; ndx++) {
				var child = obj.children[ndx];
				if (child["name"]) {
					var lvl = this.getLevelByName(child.name)
					if (lvl) {
						lvl.copy(child);
					}
				}
			}
		}
	} else {
		console.log("RetroLevelControlClass.StripeThresholdClass Error: trying to copy " + obj + " into ctrl." + this.name);
	}
}

RetroLevelControlClass.StripeThresholdClass.prototype.getLevelByName = function(name) {
	if (!name)
		return null;
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		if (this.lvlList[ndx].name.toLowerCase() == name.toLowerCase())
			return this.lvlList[ndx];
	}
	return null;
}

RetroLevelControlClass.StripeThresholdClass.prototype.getLevel4Value = function(value) {
	var lvl = null;
	if (0 < this.lvlList.length) {
		var ndx = this.lvlList.length - 1;
		lvl = this.lvlList[ndx];
		while (0 <= ndx) {
			if (lvl.minValue < value)
				break;
			ndx--;
			lvl = this.lvlList[ndx];
		}
	}
	return lvl;
}

RetroLevelControlClass.StripeThresholdClass.prototype.getHtml = function() {
	var html = "";
	html += "<span class='settings_input_label'>" + this.label + "</span>";
	html += "<table id='idTable" + this.name + "' cellspacing=0 cellpadding=0 border=0 class='threshold-table'>";
	// html += "<tr><th colspan=9 style='text-align:left;'>"
	// html += "Categories for '" + this.name + "' retro reflectance value:";
	var colWidth = Math.floor(100.0 / this.lvlList.length);
	html += "<tr>";
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		var lvl = this.lvlList[ndx];
		html += "<td class='cell" + this.name + lvl.name + " threshold-cell' style='vertical-align: top; width:" + colWidth + "%;'>";
		html += lvl.getColorSpan();
	}
	html += "<tr>";
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		var lvl = this.lvlList[ndx];
		html += "<td class='cell" + this.name + lvl.name + " threshold-cell' style='vertical-align: top; width:" + colWidth + "%;'>";
		if (0 < ndx) {
			html += lvl.getTxtCtrl();
		}
	}
	html += "</table>";
	return html;
};

RetroLevelControlClass.StripeThresholdClass.prototype.addChangeHandler = function(_callback) {
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		this.lvlList[ndx].addChangeHandler(_callback);
	}
}

RetroLevelControlClass.StripeThresholdClass.prototype.checkRange = function(level) {
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		if (this.lvlList[ndx] == level) {
			if (0 < ndx) {
				if (isNaN(level.minValue)) {
					level.minValue = this.lvlList[ndx - 1].minValue;
				}
				else if (level.minValue < this.lvlList[ndx - 1].minValue) {
					level.minValue = this.lvlList[ndx - 1].minValue;
				}
			}
			if (ndx < this.lvlList.length - 1) {
				if (isNaN(level.minValue)) {
					level.minValue = this.lvlList[ndx + 1].minValue;
				}
				else if (this.lvlList[ndx + 1].minValue < level.minValue) {
					level.minValue = this.lvlList[ndx + 1].minValue;
				}
			}
			break;
		}
	}
}

RetroLevelControlClass.StripeThresholdClass.prototype.updatePosition = function() {
	if (!RetroLevelControlClass.variableSize)
		return;
	var id = "idTable" + this.name;
	var elm = document.getElementById(id);
	if (elm) {
		ctrlWidth = 0.8 * elm.clientWidth;
	}
	var maxValue = this.maxValue;
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		if (maxValue < this.lvlList[ndx].minValue)
			maxValue = this.lvlList[ndx].minValue;
	}
	for (var ndx = 0; ndx < this.lvlList.length; ndx++) {
		var lvl = this.lvlList[ndx];
		var nextValue = maxValue;
		if (ndx < this.lvlList.length - 1) {
			nextValue = this.lvlList[ndx + 1].minValue;
		}
		var pct = (nextValue - lvl.minValue) / (maxValue / 100.0);
		pct = (pct < 0.0) ? 0.0 : pct;
		if (ndx == this.lvlList.length - 1 && pct < 1.0) {
			pct = 1.0;
		}
		var tds = document.getElementsByClassName("cell" + this.name + lvl.name);
		for (var col = 0; col < tds.length; col++) {
			tds[col].style.width = pct.toString() + "%";
			tds[col].hidden = (pct < 1) ? true : false;
		}
	}
}

// -------------------------------------------------------------------------
// level object
RetroLevelControlClass.ThresholdLevelClass = function(_parent, _name, _value, _color) {
	this.parent = _parent;
	this.name = _name;
	this.label = this.name;
	// RetroLevelControlClass["obj" + this.parent.name + this.name] = this;
	this.minValue = _value;
	this.color = new RetroLevelControlClass.RetroColorClass("Yellow", 255, 255, 0);
	if (_color)
		this.color.copy(_color);
	else if (this.name.toLowerCase() == "valid") {
		// Red "#FF0000";
		this.color.red = 255;
		this.color.green = 0;
		this.color.blue = 0;
	} else if (this.name.toLowerCase() == "good") {
		// "#00FF00"; // Lime
		this.color.red = 0;
		this.color.green = 255;
		this.color.blue = 0;
	}
	this.varname = "RetroLevelControlClass.lvl" + RetroLevelControlClass.cntr;
	RetroLevelControlClass[this.varname.substr("RetroLevelControlClass".length + 1)] = this; // eval(this.varname + " = this;");
	RetroLevelControlClass.cntr++;
}

RetroLevelControlClass.ThresholdLevelClass.prototype.copy = function(obj) {
	if (obj["min"])
		this.minValue = obj.min;
	if (obj["red"] != null && obj["green"] != null && obj["blue"] != null) {
		this.color.copy(obj);
	}
	if (obj["label"])
		this.label = obj.label;
}

RetroLevelControlClass.ThresholdLevelClass.prototype.getMin = function() {
	return this.minValue;
};

RetroLevelControlClass.ThresholdLevelClass.prototype.getColorSpan = function() {
	var html = "";
	html += "<div id='idSpan" + this.parent.name + this.name + "' style='background-color:" + this.color.toString() + "; color:" + this.color.getTextColor() + ";' class='threshold-color-span'>";
	html += this.label;
	// html += ": ";
	// html += this.minValue;
	html += "</div>";
	return html;
}
RetroLevelControlClass.ThresholdLevelClass.prototype.getTxtCtrl = function() {
	var html = "";
	html += "<input type=text style='' size='1'";
	html += " value='" + this.minValue + "'";
	html += " id='id" + this.parent.name + this.name + "' class='threshold-value-text'>";
	return html;
}

RetroLevelControlClass.ThresholdLevelClass.prototype.addChangeHandler = function(_callback) {
	this.callback = _callback;
	var input = document.getElementById("id" + this.parent.name + this.name);
	if (input) {
		input.retroLevel = this;
		input.addEventListener('change', function(evt) {
			// console.log(evt);
			evt.target.retroLevel.OnTextChange(evt.target.value);
			evt.target.value = evt.target.retroLevel.minValue.toString();
		}, false);
	}
	var span = document.getElementById("idSpan" + this.parent.name + this.name);
	if (span) {
		span.retroLevel = this;
		span.addEventListener('mouseup', this.OnMouseUp, false);
	}
}
RetroLevelControlClass.ThresholdLevelClass.prototype.OnTextChange = function(value) {
	this.minValue = parseFloat(value);
	this.parent.checkRange(this);
	this.parent.updatePosition();
	if (this.callback)
		this.callback(this);
}
RetroLevelControlClass.ThresholdLevelClass.prototype.OnMouseUp = function(evt) {
	var target = evt.target;
	var lvl = this.retroLevel;
	if (RetroLevelControlClass.lastLvl && RetroLevelControlClass.lastLvl != lvl) {
		RetroLevelControlClass.lastLvl.btnCancelHandler();
	}
	RetroLevelControlClass.lastLvl = lvl;
	var idSuffix = lvl.parent.name + lvl.name;
	lvl.span = document.getElementById("idSpan" + idSuffix);
	var html = "";
	lvl.span.style.backgroundColor = "transparent";
	lvl.span.innerHTML = "";

	var tbl = document.createElement("table");
	lvl.span.appendChild(tbl);
	var tbdy = document.createElement("tbody");
	tbl.appendChild(tbdy);
	var tr = document.createElement("tr");
	tbdy.appendChild(tr);
	var cell = document.createElement("td");
	tr.appendChild(cell);

	var divPickerWrapper = document.createElement("div");
	divPickerWrapper.setAttribute("id", "pickerWrapper" + idSuffix);
	divPickerWrapper.setAttribute("class", "picker-wrapper");
	var divPicker = document.createElement("div");
	divPicker.setAttribute("id", "picker" + idSuffix);
	divPicker.setAttribute("class", "picker");
	divPickerWrapper.appendChild(divPicker);
	var divPickerIndicator = document.createElement("div");
	divPickerIndicator.setAttribute("id", "pickerIndicator" + idSuffix);
	divPickerIndicator.setAttribute("class", "picker-indicator");
	divPickerWrapper.appendChild(divPickerIndicator);
	cell.appendChild(divPickerWrapper);

	cell = document.createElement("td");
	tr.appendChild(cell);
	var divSliderWrapper = document.createElement("div");
	divSliderWrapper.setAttribute("id", "sliderWrapper" + idSuffix);
	divSliderWrapper.setAttribute("class", "slider-wrapper");
	var divSlider = document.createElement("div");
	divSlider.setAttribute("id", "slider" + idSuffix);
	divSlider.setAttribute("class", "slider");
	divSliderWrapper.appendChild(divSlider);
	var divSliderIndicator = document.createElement("div");
	divSliderIndicator.setAttribute("id", "sliderIndicator" + idSuffix);
	divSliderIndicator.setAttribute("class", "slider-indicator");
	divSliderWrapper.appendChild(divSliderIndicator);
	cell.appendChild(divSliderWrapper);

	tr = document.createElement("tr");
	tbdy.appendChild(tr);
	var sampleCell = document.createElement("td");
	tr.appendChild(sampleCell);
	sampleCell.setAttribute("colspan", "2");
	sampleCell.setAttribute("id", "cell" + idSuffix);
	sampleCell.setAttribute("class", "color-sample-cell");
	sampleCell.appendChild(document.createTextNode("sample"));

	// RGB text
	tr = document.createElement("tr");
	tbdy.appendChild(tr);
	var txtCell = document.createElement("td");
	tr.appendChild(txtCell);
	var rgbTable = document.createElement("table");
	rgbTable.setAttribute("class", "rgb-table");
	txtCell.appendChild(rgbTable);
	var rgbTbody = document.createElement("tbody");
	rgbTable.appendChild(rgbTbody);
	var rgbNames = [ "Red", "Green", "Blue" ];
	for (var rgbNdx = 0; rgbNdx < rgbNames.length; rgbNdx++) {
		var element = rgbNames[rgbNdx];
		var rgbRow = document.createElement("tr");
		rgbTbody.appendChild(rgbRow);
		var rgbLabelCell = document.createElement("td");
		rgbLabelCell.setAttribute("class", "rgb-cell");
		rgbLabelCell.appendChild(document.createTextNode(element + ":"));
		rgbRow.appendChild(rgbLabelCell);
		var rgbTextCell = document.createElement("td");
		rgbTextCell.setAttribute("class", "rgb-cell");
		rgbRow.appendChild(rgbTextCell);
		var txt = document.createElement("input");
		txt.setAttribute("id", "txt" + element);
		txt.setAttribute("type", "text");
		txt.setAttribute("size", "4");
		txt.addEventListener('keyup', function(evt) {
			if (evt.key == "Enter") {
				// console.log("txt.keyup: " + evt.toString() );
				var rgb = {
					r : txtRed.value,
					g : txtGreen.value,
					b : txtBlue.value
				};
				lvl.cp.setRgb(rgb);
				lvl.span.style.color = lvl.color.getTextColor(rgb);
			}
		}, true);
		rgbTextCell.appendChild(txt);
	}
	// VGA colors
	var cellPreset = document.createElement("td");
	cellPreset.setAttribute("class", "rgb-cell");
	tr.appendChild(cellPreset);
	for (var vgaNdx = 0; vgaNdx < RetroLevelControlClass.vga.length; vgaNdx++) {
		var vgaColor = RetroLevelControlClass.vga[vgaNdx];
		var vgaElement = document.createElement("div");
		vgaElement.setAttribute("class", "vga-cell");
		vgaElement.appendChild(document.createTextNode(vgaColor.name));
		vgaElement.style.backgroundColor = vgaColor.toString();
		vgaElement.style.color = vgaColor.getTextColor();
		vgaElement.vgaColor = vgaColor;
		vgaElement.addEventListener('click', function(evt) {
			var elm = evt.target;
			lvl.btnSelectColorHandler(elm.vgaColor);
		}, true);
		cellPreset.appendChild(vgaElement);
	}

	// color picker
	lvl.cp = ColorPicker(divSlider, divPicker, function(hex, hsv, rgb, pickerIndicatorPosition, slideIndicatorPosition) {
		// console.log(hsv.h, hsv.s, hsv.v); // [0-359], [0-1], [0-1]
		// console.log(rgb.r, rgb.g, rgb.b); // [0-255], [0-255], [0-255]
		ColorPicker.positionIndicators(divSliderIndicator, divPickerIndicator, slideIndicatorPosition, pickerIndicatorPosition);
		// document.body.style.backgroundColor = hex; // #HEX
		sampleCell.style.backgroundColor = hex; // #HEX
		txtRed.value = rgb.r;
		txtGreen.value = rgb.g;
		txtBlue.value = rgb.b;
	});
	lvl.cp.setRgb({
		r : lvl.color.red,
		g : lvl.color.green,
		b : lvl.color.blue
	});
	this.removeEventListener('mouseup', lvl.OnMouseUp);

	tr = document.createElement("tr");
	tbdy.appendChild(tr);
	cell = document.createElement("td");
	cell.setAttribute("class", "color-button-cell");
	tr.appendChild(cell);
	cell.setAttribute("colspan", "2");
	var btnSet = document.createElement("a");
	btnSet.setAttribute("class", "color-button");
	btnSet.setAttribute("href", "javascript:" + lvl.varname + ".btnSetHandler();");
	btnSet.innerHTML = "Set";
	cell.appendChild(btnSet);
	var btnCancel = document.createElement("a");
	btnCancel.setAttribute("class", "color-button");
	btnCancel.setAttribute("href", "javascript:" + lvl.varname + ".btnCancelHandler();");
	btnCancel.innerHTML = "Cancel";
	cell.appendChild(btnCancel);
}
RetroLevelControlClass.ThresholdLevelClass.prototype.btnSelectColorHandler = function(color) {
	this.color.copy(color);
	this.closePicker();
	if (this.callback)
		this.callback(this); // send update
}
RetroLevelControlClass.ThresholdLevelClass.prototype.btnSetHandler = function() {
	// console.log(this.name + ".btnSetHandler()");
	this.cp.setRgb({
		r : txtRed.value,
		g : txtGreen.value,
		b : txtBlue.value
	});
	var rgb = ColorPicker.hsv2rgb(this.cp);
	this.color.copyRGB(rgb);
	this.closePicker();
	if (this.callback)
		this.callback(this);
}
RetroLevelControlClass.ThresholdLevelClass.prototype.btnCancelHandler = function() {
	this.closePicker();
}
RetroLevelControlClass.ThresholdLevelClass.prototype.closePicker = function() {
	this.span.style.backgroundColor = this.color.toString();
	this.span.style.color = this.color.getTextColor();
	this.span.innerHTML = this.label;
	this.span.addEventListener('mouseup', this.span.retroLevel.OnMouseUp, false);
}

RetroLevelControlClass.ThresholdLevelClass.prototype.getFields = function() { // parsed by software\Laserlux\src\threshold.c::readThreshold()
	var fields = {};
	var prefix = RetroLevelControlClass.RETRO_THRESHOLD_KEY + this.parent.name + this.name;
	fields[prefix + "min"] = this.minValue.toString();
	fields[prefix + "red"] = this.color.red.toString();
	fields[prefix + "green"] = this.color.green.toString();
	fields[prefix + "blue"] = this.color.blue.toString();
	return fields;
}

// -------------------------------------------------------------------------
// color object
RetroLevelControlClass.RetroColorClass = function(_name, _red, _green, _blue) {
	this.name = _name;
	this.red = _red != null ? _red : 0;
	this.green = _green != null ? _green : 0;
	this.blue = _blue != null ? _blue : 0;
}

RetroLevelControlClass.RetroColorClass.prototype.copy = function(obj) {
	this.red = obj.red;
	this.green = obj.green;
	this.blue = obj.blue;
}

RetroLevelControlClass.RetroColorClass.prototype.copyRGB = function(rgb) {
	this.red = rgb.r;
	this.green = rgb.g;
	this.blue = rgb.b;
}

RetroLevelControlClass.RetroColorClass.prototype.asRGB = function() {
	return {
		r : this.red,
		g : this.green,
		b : this.blue
	};
}

RetroLevelControlClass.RetroColorClass.prototype.toString = function() {
	var value = "rgb(" + this.red.toString() + "," + this.green.toString() + "," + this.blue.toString() + ")";
	return value;
}
RetroLevelControlClass.fgColorLimit = 128;
RetroLevelControlClass.RetroColorClass.prototype.getTextColor = function(rgb) {
	if (rgb == null) {
		rgb = {
			r : this.red,
			g : this.green,
			b : this.blue
		};
	}
	var colorname = "Black";
	var colorsum = Number(rgb.r) + Number(rgb.g) + Number(rgb.b);
	if (colorsum < 3 * RetroLevelControlClass.fgColorLimit)
		colorname = "White";
	return colorname;
}

RetroLevelControlClass.Black = new RetroLevelControlClass.RetroColorClass("Black");

// -------------------------------------------------------------------------
// initialize this module
RetroLevelControlClass.init();
