// StripChartClass.js

var StripChartClass = {
	name : "StripChartClass",
	description : "namespace for Gamma-Scientific LaserLux_G7 strip chart display",
	version : '1.0',
	miny : 0.0,
	maxy : 100.0,
	minx : 0.0,
	maxx : 1.0,
	stripeColorNdx : 0,
	data : []
};
StripChartClass.RETRO_STRIP_CHART_DATA_KEY = "RSCD";
StripChartClass.STRIPCHART_LENGTH = 450;
StripChartClass.labelInterval = 100
StripChartClass.labelMargin = 5
StripChartClass.dumpdata = false;

// -------------------------------------------------------------------------
// static functions
StripChartClass.init = function() {
	this.data = []; // clear list
}

StripChartClass.setStripColorNdx = function(value) {
	this.stripeColorNdx = value;
}

StripChartClass.readProp = function(value) {
	if (value && 0 < value.length) {
		this.extractData(value);
		this.refreshChart();
		if (this.dumpdata) {
			console.log("StripChartClass.data.length=" + this.data.length);
			for (var ndx = 0; ndx < this.data.length; ndx++) {
				var pt = this.data[ndx];
				console.log(pt.odometer.toString() + "= " + pt.left.toString() + ", " + pt.right.toString());
			}
			this.dumpdata = false;
		}
	}
}

StripChartClass.extractData = function(value) {
	if (!value)
		return;
	this.data = []; // clear list
	this.minx = value[0].odometer;
	this.maxx = value[0].odometer;
	for (var ndx = 0; ndx < value.length; ndx++) {
		this.data[this.data.length] = value[ndx];
		this.maxx = value[ndx].odometer;
		if (this.STRIPCHART_LENGTH <= this.data.length)
			break;
	}
}

StripChartClass.getCtx = function() {
	var canvas = document.getElementById("idStripChart");
	if (canvas) {
		canvas.height = canvas.clientHeight;
		canvas.width = canvas.clientWidth;
		this.height = canvas.height;
		this.width = canvas.width;
		var ctx = canvas.getContext("2d");
		ctx.clearRect(0, 0, canvas.width, canvas.height);
		return ctx;
	}
	return null;
}

StripChartClass.refreshChart = function() {
	var ctx = this.getCtx();
	this.drawThresholds(ctx);
	this.plotData(ctx, "left");
	if (this.stripeTypeId == "AST2" || this.stripeTypeId == "AST3" || this.stripeTypeId == "AST4") {
		this.plotData(ctx, "right");
	}
	this.plotLabels(ctx);
}

StripChartClass.drawThresholds = function(ctx) {
	if (!ctx) {
		ctx = this.getCtx();
		if (!ctx)
			return;
	}
	var stripe = RetroLevelControlClass.getStripe(this.stripeColorNdx);
	if (stripe) {
		// set chart scale
		this.maxy = y_axis_max < rpm ? y_axis_max : rpm; // stripe.maxValue;
		// draw thresholds
		var level = stripe.lvlList[2]; // RetroLevelControlClass.ThresholdLevelClass
		ctx.beginPath();
		ctx.strokeStyle = level.color.toString();
		var pt = this.translatePoint(this.minx, level.minValue);
		ctx.moveTo(pt.x, pt.y);
		var pt = this.translatePoint(this.maxx, level.minValue);
		ctx.lineTo(pt.x, pt.y);
		ctx.stroke();
	}
}

StripChartClass.plotData = function(ctx, channel) {
	if (!ctx) {
		ctx = this.getCtx();
		if (!ctx)
			return;
	}
	if (this.data && 0 < this.data.length) {
		if (this.data[0][channel] == undefined)
			return;
		var stripe = RetroLevelControlClass.getStripe(this.stripeColorNdx);
		var pt = this.translatePoint(this.data[0].odometer, this.data[0][channel]);
		for (var ndx = 1; ndx < this.data.length; ndx++) {
			ctx.beginPath();
			ctx.moveTo(pt.x, pt.y);
			pt = this.translatePoint(this.data[ndx].odometer, this.data[ndx][channel]);
			var lvl = stripe.getLevel4Value(this.data[ndx][channel]);
			if (lvl != null) {
				ctx.strokeStyle = lvl.color.toString();
				ctx.lineTo(pt.x, pt.y);
				ctx.stroke();
			}
		}
		ctx.font = '10px monospace';
		ctx.fillStyle = stripe.bckcolor;
		var firstPt = this.data[0];
		var pt = this.translatePoint(firstPt.odometer, firstPt[channel]);
		ctx.fillText(firstPt[channel].toFixed(1), pt.x, pt.y);
	}
}

StripChartClass.plotLabels = function(ctx) {
	if (this.data && 0 < this.data.length) {
		if (!ctx) {
			ctx = this.getCtx();
			if (!ctx)
				return;
		}
		var stripe = RetroLevelControlClass.getStripe(this.stripeColorNdx);
		ctx.font = '10px monospace';
		ctx.fillStyle = stripe.bckcolor;
		var min_odo = this.data[0].odometer;
		var max_odo = this.data[this.data.length - 1].odometer;
		var range = Math.abs(max_odo - min_odo);
		var resolution = (100.0 < range) ? 10.0 : (10.0 < range) ? 1.0 : 0.1;
		var num_decimals = resolution < 1.0 ? 1 : 0;
		var last_odo = Math.floor(this.data[0].odometer / resolution);
		var ndx = 1;
		while (ndx < this.data.length) {
			var this_odo = Math.floor(this.data[ndx].odometer / resolution);
			if (last_odo != this_odo) {
				last_odo = this_odo;
				var pt = this.translatePoint(this.data[ndx].odometer, 0);
				ctx.fillText(this.data[ndx].odometer.toFixed(num_decimals), pt.x, this.height - this.labelMargin);
				ndx += this.labelInterval / 2.0;
			}
			ndx++;
		}
	}
}

StripChartClass.translatePoint = function(value_x, value_y) {
	var chart_x = (value_x - this.minx) / (this.maxx - this.minx) * this.width;
	var chart_y = this.height - (value_y - this.miny) / (this.maxy - this.miny) * this.height;
	return {
		x : chart_x,
		y : chart_y
	};
}

StripChartClass.clickedOnChart = function(event) {
	var newminy = 0.0;
	var stripe = RetroLevelControlClass.getStripe(this.stripeColorNdx);
	for (var ndx = 0; ndx < stripe.lvlList.length; ndx++) {
		var level = stripe.lvlList[ndx]; // RetroLevelControlClass.ThresholdLevelClass
		if (this.miny < level.minValue) {
			newminy = level.minValue;
			console.log(" set miny to " + level.parent.name + "." + level.name + ".minValue = " + newminy);
			break;
		}
	}
	this.miny = newminy;
	this.refreshChart();
}

// initialize
StripChartClass.init();
